﻿using System;
using Crestron.SimplSharp;                          				// For Basic SIMPL# Classes
using Crestron.SimplSharpPro;                       				// For Basic SIMPL#Pro classes
using System.Text.RegularExpressions;
using Crestron.SimplSharp.CrestronSockets;
using Crestron.SimplSharpPro.CrestronThread;

namespace Inogeni
{
    public class CAM300
    {
        public event EventHandler<PropertyChangedEventArgs> PropertyChanged;

        private string _ipaddress;
        public string IPAddress
        {
            get { return (_ipaddress); }
            set
            {
                _ipaddress = value;
                InvokePropertyChanged("IPAddress", _ipaddress);
            }
        }

        private string _macaddress;
        public string MACAddress
        {
            get
            {
                return _macaddress;
            }
            set
            {
                _macaddress = value;
                InvokePropertyChanged("MACAddress", _macaddress);
            }
        }

        private string _subnetmask;
        public string SubnetMask
        {
            get
            {
                return _subnetmask;
            }
            set
            {
                _subnetmask = value;
                InvokePropertyChanged("SubnetMask", _subnetmask);
            }
        }

        //continue with this event invocation later...

        public void SimulateResponse(string input)
        {
            if (input.Length > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + input;
            }
        }

        public ConnectionType ConnectionType;
        public ComPort SerialPort;
        TCPClient Client;

        CTimer PollTimer;
        bool PollTimerIsTicking;

        bool _enableregularpolling;
        public bool EnableRegularPolling
        {
            get
            {
                return _enableregularpolling;
            }
            set
            {
                switch (value)
                {
                    case(true):
                    {
                        if (RegularPollInterval > 0)
                        {
                            if (_enableregularpolling == false)
                            {
                                _enableregularpolling = true;
                                RegularPoll(_enableregularpolling);
                            }
                        }
                        break;
                    }
                    default:
                    {
                        _enableregularpolling = false;
                        PollTimerIsTicking = false;
                        if (PollTimer != null)
                        {
                            PollTimer.Stop();
                            PollTimer.Dispose();
                        }
                        break; 
                    }
                }
            }
        }

        public long RegularPollInterval = 60000;
        string _incomingdatabuffer;

        public string _streamerappversion;
        public string StreamerAppVersion
        {
            get
            {
                return _streamerappversion;
            }
            private set
            {
                _streamerappversion = value;
                InvokePropertyChanged("StreamerApp", value);
            }
            
        }

        public ushort DeviceID;
        public string FX3Version;
        public string FPGAVersion;
        public string Keypad;

        bool _blackoutstate;
        public bool BlackoutState
        {
            get
            {
                return _blackoutstate;
            }
            private set
            {
                _blackoutstate = value;
                InvokePropertyChanged("Blackout", value);
            }
        }

        public uint[] AIN;
        public uint AOUT;
        public uint ACFG;

        uint[] _zoomfactor;
        public PropertyArray<uint> ZoomFactor;


        InversionState[] _setinversionstate;
        InversionState[] _inversionstatefb;

        public PropertyArray<InversionState> SetInversionState;
        public PropertyArray<InversionState> InversionStateFB;

        public bool IsConnected { get; private set; }

        string IncomingDataBuffer
        {
            get
            {
                return _incomingdatabuffer;
            }
            set
            {
                _incomingdatabuffer = value;
                Parse();
            }
        }

        char StartPrefix;
        char EndPrefix;
        char Separator;

        public event EventHandler<DebugEventArgs> DebugInfo;
        void Debug(string info)
        {
            if (DebugInfo != null)
            {
                DebugInfo(this, new DebugEventArgs(info));
            }
        }

        static int Port = 50000;

        ushort _currentsource;

        public ushort CurrentInput
        {
            get
            {
                return _currentsource;
            }
            private set
            {
                InvokePropertyChanged("CurrentSource", value);
                _currentsource = value;
            }
        }

        public PropertyArray<string> CurrentResolutions;

        string [] _currentresolutions;

        void ConfigureSerialPort(ref ComPort port)
        {
            SerialPort = port;
            SerialPort.SetComPortSpec(ComPort.eComBaudRates.ComspecBaudRate9600,
                ComPort.eComDataBits.ComspecDataBits8,
                ComPort.eComParityType.ComspecParityNone,
                ComPort.eComStopBits.ComspecStopBits1,
                ComPort.eComProtocolType.ComspecProtocolRS232,
                ComPort.eComHardwareHandshakeType.ComspecHardwareHandshakeNone,
                ComPort.eComSoftwareHandshakeType.ComspecSoftwareHandshakeNone,
                false);
        }

        void InitializeArrays()
        {
            AIN = new uint[5];
            _setinversionstate = new InversionState[5];
            _inversionstatefb = new InversionState[5];

            _currentresolutions = new string[5];
            CurrentResolutions = new PropertyArray<string>(_currentresolutions);

            SetInversionState = new PropertyArray<InversionState>(_setinversionstate);
            InversionStateFB = new PropertyArray<InversionState>(_inversionstatefb);

            _zoomfactor = new uint[5] {100,100,100,100,100};
            ZoomFactor = new PropertyArray<uint>(_zoomfactor);

            ZoomFactor.ArrayChanged += ZoomFactorArrayChanged;

            SetInversionState.ArrayChanged += SetInversionStateArrayChanged;
            InversionStateFB.ArrayChanged += InversionStateFBArrayChanged;

            CurrentResolutions.ArrayChanged += CurrentResolutionArrayChanged;
        }

        public CAM300(ref ComPort serialPort)
        {
            ConnectionType = ConnectionType.RS232;
            try
            {
                ConfigureSerialPort(ref serialPort);
                IsConnected = true;
            }
            catch
            {
                IsConnected = false;
                Debug("Error initializing serial port.");
            }
            StartPrefix = '<';
            EndPrefix = '>';
            Separator = ' ';

            InitializeArrays();

            SerialPort.SerialDataReceived += OnSerialDataReceived;
        }

        void RegularPoll(object obj)
        {
            PollTimerIsTicking = false;
            if (obj.GetType() == typeof(bool))
            {
                if (Convert.ToBoolean(obj) == true)
                {
                    Get();
                    if (PollTimer != null)
                    {
                        PollTimer.Dispose();
                    }
                    PollTimer = new CTimer(RegularPoll, _enableregularpolling, RegularPollInterval);
                    PollTimerIsTicking = true;
                }
            }
        }

        void OnSerialDataReceived(ComPort ReceivingComPort, ComPortSerialDataEventArgs args)
        {
            if (args.SerialData.Length > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + args.SerialData;
            }
        }
        void OnIPDataReceived(TCPClient client, int numBytesReceived)
        {
            if (numBytesReceived > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + System.Text.Encoding.UTF8.GetString(client.IncomingDataBuffer, 0, numBytesReceived);
                if (Client.ClientStatus == SocketStatus.SOCKET_STATUS_CONNECTED)
                {
                    Client.ReceiveDataAsync(OnIPDataReceived);
                }
            }
        }

        public CAM300(string IPAddress)
        {
            ConnectionType = ConnectionType.IP;
            Client = new TCPClient(IPAddress, Port, 65535);
            StartPrefix = '$';
            EndPrefix = '$';
            Separator = '_';
            InitializeArrays();
            Client.SocketStatusChange += OnConnectionStatusChange;
            Connect();
        }

        public CAM300(IPAddress Address)
        {
            ConnectionType = ConnectionType.IP;
            Client = new TCPClient(new IPEndPoint(Address, Port), 65535);
            StartPrefix = '$';
            EndPrefix = '$';
            Separator = '_';
            InitializeArrays();
            Client.SocketStatusChange += OnConnectionStatusChange;
            Connect();
        }

        public SocketErrorCodes Connect()
        {
            if (Client != null)
            {
                return Client.ConnectToServer();
            }
            else
            {
                ErrorLog.Notice("Connect method not required for RS232 devices.");
                return SocketErrorCodes.SOCKET_BUFFER_NOT_ALLOCATED;
            }
        }

        public SocketErrorCodes Disconnect()
        {
            if (Client != null)
            {
                SocketErrorCodes DisconnectCode = Client.DisconnectFromServer();
                Client.Dispose();
                return DisconnectCode;
            }
            else
            {
                ErrorLog.Notice("Disconnect not used for RS232 devices.");
                return SocketErrorCodes.SOCKET_BUFFER_NOT_ALLOCATED;
            }
        }

        void OnConnect()
        {
            Client.ReceiveDataAsync(OnIPDataReceived);
            IsConnected = true;
        }

        void OnDisconnect()
        {
            IsConnected = false;
        }

        void OnConnectionStatusChange(TCPClient Client, SocketStatus status)
        {
            if (status == SocketStatus.SOCKET_STATUS_CONNECTED)
            {
                OnConnect();
            }
            else
            {
                OnDisconnect();
            }
        }

        public void Get()
        {
            SendCommand("GET");
            if (PollTimerIsTicking)
            {
                PollTimer.Reset(RegularPollInterval);
                //ErrorLog.Notice("poll timer reset.");
            }
        }
        public void Poll()
        {
            SendCommand("POLL");
        }
        public void SetInput(ushort sourceSelected)
        {
            SendCommand("IN" + sourceSelected.ToString());
            Thread T = new Thread(GetAsync, null);
        }
        public void Blackout()
        {
            SendCommand("BLK");
            Thread T = new Thread(GetAsync, null); //poll after sending command if RS232. IP gives auto update on all changes.
        }
        public void UnBlackout()
        {
            SendCommand("SHW");
            Thread T = new Thread(GetAsync, null); //poll after sending command if RS232. IP gives auto update on all changes.
        }
        public void Reset()
        {
            SendCommand("RST");
        }

        void Pan(ushort camNumber, short panDirection)
        {
            SendCommand("PAN" + Separator + camNumber.ToString() + Separator + panDirection.ToString());
        }

        public void PanCounterClockwise(ushort camNumber)
        {
            Pan(camNumber, -1);
        }
        public void PanClockwise(ushort camNumber)
        {
            Pan(camNumber, 1);
        }
        public void StopPan(ushort camNumber)
        {
            Pan(camNumber, 0);
        }

        void Tilt(ushort camNumber, short TiltDirection)
        {
            SendCommand("TILT" + Separator + camNumber.ToString() + Separator + TiltDirection.ToString());
        }

        public void TiltDown(ushort camNumber)
        {
            Tilt(camNumber, -1);
        }
        public void TiltUp(ushort camNumber)
        {
            Tilt(camNumber, 1);
        }
        public void StopTilt(ushort camNumber)
        {
            Tilt(camNumber, 0);
        }

        public void SetZoomValue(ushort camNumber, ushort zoomFactor)
        {
            SendCommand("ZOOM" + Separator + camNumber.ToString() + Separator + zoomFactor.ToString());
        }
        public void ZoomIn(ushort camNumber, uint stepSize)
        {
            if ((ZoomFactor[camNumber] + stepSize) <= 500)
            {
                ZoomFactor[camNumber] = ZoomFactor[camNumber] + stepSize;
            }
        }
        public void ZoomOut(ushort camNumber, uint stepSize)
        {
            if ((ZoomFactor[camNumber] - stepSize) >= 100)
            {
                ZoomFactor[camNumber] = ZoomFactor[camNumber] - stepSize;
            }
        }

        void SendCommand(string Command)
        {
            if (this.ConnectionType == ConnectionType.RS232)
            {
                try
                {
                    SerialPort.Send(StartPrefix + Command + EndPrefix);
                }
                catch (ArgumentNullException)
                {
                    if (SerialPort == null)
                    {
                        Debug("Serial port used in initialization was null.");
                    }
                    if (Command == null)
                    {
                        Debug("Text passed to serial port was null.");
                    }
                }
            }
            else if (this.ConnectionType == ConnectionType.IP)
            {
                byte[] toSend = System.Text.Encoding.UTF8.GetBytes(StartPrefix + Command + EndPrefix);
                try
                {
                    Client.SendData(toSend, toSend.Length);
                }
                catch (Exception e)
                {
                    ErrorLog.Notice("Error sending data via IP: {0}", e.Message);
                }
            }
        }

        void Parse()
        {
            while (IncomingDataBuffer.Contains("\x0A"))
            {
                string toParse = IncomingDataBuffer.Substring(0, IncomingDataBuffer.IndexOf("\n") + 1);
                _incomingdatabuffer = _incomingdatabuffer.Remove(_incomingdatabuffer.IndexOf(toParse), toParse.Length);
                Regex ToMatch = new Regex(@"(?<name>.+)\s=>\s(?<value>[^\n\r]+)[\n\r\s]+");
                Match M = ToMatch.Match(toParse);
                //ErrorLog.Notice(toParse);
                if (M.Success)
                {
                    switch (M.Groups["name"].Value)
                    {
                        case ("StreamerApp"):
                            {
                                StreamerAppVersion = M.Groups["value"].Value;
                                break;
                            }
                        case ("DeviceID"):
                            {
                                DeviceID = Convert.ToUInt16(M.Groups["value"].Value);
                                break;
                            }
                        case ("FX3"):
                            {
                                FX3Version = M.Groups["value"].Value;
                                break;
                            }
                        case ("FPGA"):
                            {
                                FPGAVersion = M.Groups["value"].Value;
                                break;
                            }
                        case ("KEYPAD"):
                            {
                                Keypad = M.Groups["value"].Value;
                                break;
                            }
                        case ("BLACK"):
                            {
                                if (M.Groups["value"].Value.Contains("disabled"))
                                {
                                    BlackoutState = false;
                                }
                                else
                                {
                                    BlackoutState = true;
                                }
                                break;
                            }
                        case ("IP"):
                            {
                                IPAddress = M.Groups["value"].Value;
                                break;
                            }
                        case ("MAC"):
                            {
                                MACAddress = M.Groups["value"].Value;
                                break;
                            }
                        case ("NETMASK"):
                            {
                                SubnetMask = M.Groups["value"].Value;
                                break;
                            }
                        case ("VIEW"):
                            {
                                CurrentInput = Convert.ToUInt16(M.Groups["value"].Value.Substring(2, 1));
                                break;
                            }
                        case ("ACFG"):
                            {
                                ACFG = Convert.ToUInt16(M.Groups["value"].Value);
                                break;
                            }
                        default: // returns that have numbers in the response type that need parsing...
                            {
                                if (M.Groups["name"].Value.Contains("Input ") && M.Groups["name"].Value.Length == 7)
                                {
                                    CurrentResolutions[Convert.ToUInt16(M.Groups["name"].Value.Substring(6,1))] = M.Groups["value"].Value;
                                }
                                else if (M.Groups["name"].Value.Contains("AIN") && M.Groups["name"].Value.Length == 4)
                                {
                                    AIN[Convert.ToUInt16(M.Groups["name"].Value.Substring(3,1))] = Convert.ToUInt16(M.Groups["value"].Value);
                                }
                                else if (M.Groups["name"].Value.Contains("AOUT") && M.Groups["name"].Value.Length == 5)
                                {
                                    AOUT = Convert.ToUInt16(M.Groups["value"].Value);
                                }
                                else if (M.Groups["name"].Value.Contains("INV") && M.Groups["name"].Value.Length == 4)
                                {
                                    InversionStateFB[Convert.ToUInt16(M.Groups["name"].Value.Substring(3,1))] = GetInversionState(Convert.ToUInt16((M.Groups["value"].Value)));
                                }
                                break;
                            }
                    }
                }
                else if (toParse == "ACK\x0D\x0A")
                {
                    //Get(); // some command succeeded.
                }
                else if (toParse == "NACK\x0D\x0A")
                {
                    //Get(); // some command failed.
                }
                if (_incomingdatabuffer.Length > 1000)
                {
                    _incomingdatabuffer = "";
                    ErrorLog.Notice("Had to clear buffer...");
                }
            }
        }

        InversionState GetInversionState(ushort value)
        {
            switch (value)
            {
                case (1): { return InversionState.VerticalFlip; }
                case (2): { return InversionState.HorizontalFlip; }
                default: { return InversionState.NoFlip; }
            }
        }

        ushort GetInversionValue(InversionState s)
        {
            switch (s)
            {
                case (InversionState.NoFlip):
                {
                    return 0;
                }
                case(InversionState.HorizontalFlip):
                {
                    return 2;
                }
                case (InversionState.VerticalFlip):
                    {
                        return 1;
                    }
                default: { return 0; }
            }
        }

        void InvokePropertyChanged(string name, object value)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(name, value));
            }
        }

        void SetInversionStateArrayChanged(object sender, ArrayChangedEventArgs<InversionState> args)
        {
            SendCommand("INV" + Separator + args.index.ToString() + Separator + GetInversionValue(args.value).ToString());
            Thread T = new Thread(GetAsync, null);
        }

        void InversionStateFBArrayChanged(object sender, ArrayChangedEventArgs<InversionState> args)
        {
            PropertyChanged(this, new PropertyChangedEventArgs("InversionStateFB[" + args.index.ToString() + "]", GetInversionValue(args.value).ToString()));
        }
        void CurrentResolutionArrayChanged(object sender, ArrayChangedEventArgs<string> args)
        {
            PropertyChanged(this, new PropertyChangedEventArgs("CurrentResolutions[" + args.index.ToString() + "]", args.value.ToString()));
        }

        void ZoomFactorArrayChanged(object sender, ArrayChangedEventArgs<uint> args)
        {
            _zoomfactor[args.index] = args.value;
            SetZoomValue((ushort)args.index, (ushort)args.value);
        }

        object GetAsync(object o)
        {
            Crestron.SimplSharpPro.CrestronThread.Thread.Sleep(500);
            Get();
            return null;
        }

    }
    public class SHARE2U
    {
        public event EventHandler<PropertyChangedEventArgs> PropertyChanged;

        private string _ipaddress;
        public string IPAddress
        {
            get { return (_ipaddress); }
            set
            {
                _ipaddress = value;
                InvokePropertyChanged("IPAddress", _ipaddress);
            }
        }

        private string _macaddress;
        public string MACAddress
        {
            get
            {
                return _macaddress;
            }
            set
            {
                _macaddress = value;
                InvokePropertyChanged("MACAddress", _macaddress);
            }
        }

        private string _subnetmask;
        public string SubnetMask
        {
            get
            {
                return _subnetmask;
            }
            set
            {
                _subnetmask = value;
                InvokePropertyChanged("SubnetMask", _subnetmask);
            }
        }

        //continue with this event invocation later...

        public void SimulateResponse(string input)
        {
            if (input.Length > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + input;
            }
        }

        public ConnectionType ConnectionType;
        public ComPort SerialPort;
        TCPClient Client;

        CTimer PollTimer;
        bool PollTimerIsTicking;

        bool _enableregularpolling;
        public bool EnableRegularPolling
        {
            get
            {
                return _enableregularpolling;
            }
            set
            {
                switch (value)
                {
                    case(true):
                    {
                        if (RegularPollInterval > 0)
                        {
                            if (_enableregularpolling == false)
                            {
                                _enableregularpolling = true;
                                RegularPoll(_enableregularpolling);
                            }
                        }
                        break;
                    }
                    default:
                    {
                        _enableregularpolling = false;
                        PollTimerIsTicking = false;
                        if (PollTimer != null)
                        {
                            PollTimer.Stop();
                            PollTimer.Dispose();
                        }
                        break; 
                    }
                }
            }
        }

        public long RegularPollInterval = 60000;
        string _incomingdatabuffer;

        public string _streamerappversion;
        public string StreamerAppVersion
        {
            get
            {
                return _streamerappversion;
            }
            private set
            {
                _streamerappversion = value;
                InvokePropertyChanged("StreamerApp", value);
            }
            
        }

        public ushort DeviceID;
        public string FX3Version;
        public string FPGAVersion;
        public string Keypad;

        bool _blackoutstate;
        public bool BlackoutState
        {
            get
            {
                return _blackoutstate;
            }
            private set
            {
                _blackoutstate = value;
                InvokePropertyChanged("Blackout", value);
            }
        }

        public sbyte[] AIN;
        public sbyte[] AOUT;
        public uint ACFG;

        uint[] _zoomfactor;
        public PropertyArray<uint> ZoomFactor;


        InversionState[] _setinversionstate;
        InversionState[] _inversionstatefb;

        public PropertyArray<InversionState> SetInversionState;
        public PropertyArray<InversionState> InversionStateFB;

        public bool IsConnected { get; private set; }

        string IncomingDataBuffer
        {
            get
            {
                return _incomingdatabuffer;
            }
            set
            {
                _incomingdatabuffer = value;
                Parse();
            }
        }

        char StartPrefix;
        char EndPrefix;
        char Separator;

        public event EventHandler<DebugEventArgs> DebugInfo;
        void Debug(string info)
        {
            if (DebugInfo != null)
            {
                DebugInfo(this, new DebugEventArgs(info));
            }
        }

        static int Port = 50000;

        Layout _currentlayout;

        public Layout CurrentLayout
        {
            get
            {
                return _currentlayout;
            }
            private set
            {
                InvokePropertyChanged("CurrentLayout", value);
                _currentlayout = value;
            }
        }

        public PropertyArray<string> CurrentResolutions;

        string [] _currentresolutions;

        void ConfigureSerialPort(ref ComPort port)
        {
            SerialPort = port;
            SerialPort.SetComPortSpec(ComPort.eComBaudRates.ComspecBaudRate9600,
                ComPort.eComDataBits.ComspecDataBits8,
                ComPort.eComParityType.ComspecParityNone,
                ComPort.eComStopBits.ComspecStopBits1,
                ComPort.eComProtocolType.ComspecProtocolRS232,
                ComPort.eComHardwareHandshakeType.ComspecHardwareHandshakeNone,
                ComPort.eComSoftwareHandshakeType.ComspecSoftwareHandshakeNone,
                false);
        }

        void InitializeArrays()
        {
            AIN = new sbyte[5];
            AOUT = new sbyte[5];
            _setinversionstate = new InversionState[5];
            _inversionstatefb = new InversionState[5];

            _currentresolutions = new string[5];
            CurrentResolutions = new PropertyArray<string>(_currentresolutions);

            SetInversionState = new PropertyArray<InversionState>(_setinversionstate);
            InversionStateFB = new PropertyArray<InversionState>(_inversionstatefb);

            _zoomfactor = new uint[5] {100,100,100,100,100};
            ZoomFactor = new PropertyArray<uint>(_zoomfactor);

            ZoomFactor.ArrayChanged += ZoomFactorArrayChanged;

            SetInversionState.ArrayChanged += SetInversionStateArrayChanged;
            InversionStateFB.ArrayChanged += InversionStateFBArrayChanged;

            CurrentResolutions.ArrayChanged += CurrentResolutionArrayChanged;
        }

        public SHARE2U(ref ComPort serialPort)
        {
            ConnectionType = ConnectionType.RS232;
            try
            {
                ConfigureSerialPort(ref serialPort);
                IsConnected = true;
            }
            catch
            {
                IsConnected = false;
                Debug("Error initializing serial port.");
            }
            StartPrefix = '<';
            EndPrefix = '>';
            Separator = ' ';

            InitializeArrays();

            SerialPort.SerialDataReceived += OnSerialDataReceived;
        }

        void RegularPoll(object obj)
        {
            PollTimerIsTicking = false;
            if (obj.GetType() == typeof(bool))
            {
                if (Convert.ToBoolean(obj) == true)
                {
                    Get();
                    if (PollTimer != null)
                    {
                        PollTimer.Dispose();
                    }
                    PollTimer = new CTimer(RegularPoll, _enableregularpolling, RegularPollInterval);
                    PollTimerIsTicking = true;
                }
            }
        }

        void OnSerialDataReceived(ComPort ReceivingComPort, ComPortSerialDataEventArgs args)
        {
            if (args.SerialData.Length > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + args.SerialData;
            }
        }
        void OnIPDataReceived(TCPClient client, int numBytesReceived)
        {
            if (numBytesReceived > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + System.Text.Encoding.UTF8.GetString(client.IncomingDataBuffer, 0, numBytesReceived);
                if (Client.ClientStatus == SocketStatus.SOCKET_STATUS_CONNECTED)
                {
                    Client.ReceiveDataAsync(OnIPDataReceived);
                }
            }
        }

        public SHARE2U(string IPAddress)
        {
            ConnectionType = ConnectionType.IP;
            Client = new TCPClient(IPAddress, Port, 65535);
            StartPrefix = '$';
            EndPrefix = '$';
            Separator = '_';
            InitializeArrays();
            Client.SocketStatusChange += OnConnectionStatusChange;
            Connect();
        }

        public SHARE2U(IPAddress Address)
        {
            ConnectionType = ConnectionType.IP;
            Client = new TCPClient(new IPEndPoint(Address, Port), 65535);
            StartPrefix = '$';
            EndPrefix = '$';
            Separator = '_';
            InitializeArrays();
            Client.SocketStatusChange += OnConnectionStatusChange;
            Connect();
        }

        public SocketErrorCodes Connect()
        {
            if (Client != null)
            {
                return Client.ConnectToServer();
            }
            else
            {
                ErrorLog.Notice("Connect method not required for RS232 devices.");
                return SocketErrorCodes.SOCKET_BUFFER_NOT_ALLOCATED;
            }
        }

        public SocketErrorCodes Disconnect()
        {
            if (Client != null)
            {
                SocketErrorCodes DisconnectCode = Client.DisconnectFromServer();
                Client.Dispose();
                return DisconnectCode;
            }
            else
            {
                ErrorLog.Notice("Disconnect not used for RS232 devices.");
                return SocketErrorCodes.SOCKET_BUFFER_NOT_ALLOCATED;
            }
        }

        void OnConnect()
        {
            Client.ReceiveDataAsync(OnIPDataReceived);
            IsConnected = true;
        }

        void OnDisconnect()
        {
            IsConnected = false;
        }

        void OnConnectionStatusChange(TCPClient Client, SocketStatus status)
        {
            if (status == SocketStatus.SOCKET_STATUS_CONNECTED)
            {
                OnConnect();
            }
            else
            {
                OnDisconnect();
            }
        }

        public void Get()
        {
            SendCommand("GET");
            if (PollTimerIsTicking)
            {
                PollTimer.Reset(RegularPollInterval);
                //ErrorLog.Notice("poll timer reset.");
            }
        }
        public void Poll()
        {
            SendCommand("POLL");
        }
        public void SetLayout(Layout layout)
        {
            SendCommand(layout.ToString());
            Thread T = new Thread(GetAsync, null);
        }
        public void SwapVideo()
        {
            SendCommand("SW");
            Thread T = new Thread(GetAsync, null);
        }
        public void Blackout()
        {
            SendCommand("BLK");
            Thread T = new Thread(GetAsync, null);
        }
        public void UnBlackout()
        {
            SendCommand("SHW");
            Thread T = new Thread(GetAsync, null);
        }
        public void Reset()
        {
            SendCommand("RST");
        }

        public void SetAIN(byte input, sbyte value)
        {
            SendCommand("AIN" + input.ToString() + Separator + Utils.TwosComplement(value));
            Thread T = new Thread(GetAsync, null);
        }

        public void SetAOUT(byte output, sbyte value)
        {
            SendCommand("AOUT" + output.ToString() + Separator + Utils.TwosComplement(value));
            Thread T = new Thread(GetAsync, null);
        }

        public void SetUSBAsInput2()
        {
            SendCommand("SWUSB");
            Thread T = new Thread(GetAsync, null);
        }
        public void SetHDMIAsInput2()
        {
            SendCommand("SWHDMI");
            Thread T = new Thread(GetAsync, null);
        }

        void Pan(ushort camNumber, short panDirection)
        {
            SendCommand("PAN" + Separator + camNumber.ToString() + Separator + panDirection.ToString());
        }

        public void PanCounterClockwise(ushort camNumber)
        {
            Pan(camNumber, -1);
        }
        public void PanClockwise(ushort camNumber)
        {
            Pan(camNumber, 1);
        }
        public void StopPan(ushort camNumber)
        {
            Pan(camNumber, 0);
        }

        void Tilt(ushort camNumber, short TiltDirection)
        {
            SendCommand("TILT" + Separator + camNumber.ToString() + Separator + TiltDirection.ToString());
        }

        public void TiltDown(ushort camNumber)
        {
            Tilt(camNumber, -1);
        }
        public void TiltUp(ushort camNumber)
        {
            Tilt(camNumber, 1);
        }
        public void StopTilt(ushort camNumber)
        {
            Tilt(camNumber, 0);
        }

        public void SetZoomValue(ushort camNumber, ushort zoomFactor)
        {
            SendCommand("ZOOM" + Separator + camNumber.ToString() + Separator + zoomFactor.ToString());
        }
        public void ZoomIn(ushort camNumber, uint stepSize)
        {
            if ((ZoomFactor[camNumber] + stepSize) <= 500)
            {
                ZoomFactor[camNumber] = ZoomFactor[camNumber] + stepSize;
            }
        }
        public void ZoomOut(ushort camNumber, uint stepSize)
        {
            if ((ZoomFactor[camNumber] - stepSize) >= 100)
            {
                ZoomFactor[camNumber] = ZoomFactor[camNumber] - stepSize;
            }
        }

        void SendCommand(string Command)
        {
            if (this.ConnectionType == ConnectionType.RS232)
            {
                try
                {
                    SerialPort.Send(StartPrefix + Command + EndPrefix);
                }
                catch (ArgumentNullException)
                {
                    if (SerialPort == null)
                    {
                        Debug("Serial port used in initialization was null.");
                    }
                    if (Command == null)
                    {
                        Debug("Text passed to serial port was null.");
                    }
                }
            }
            else if (this.ConnectionType == ConnectionType.IP)
            {
                byte[] toSend = System.Text.Encoding.UTF8.GetBytes(StartPrefix + Command + EndPrefix);
                try
                {
                    Client.SendData(toSend, toSend.Length);
                }
                catch (Exception e)
                {
                    ErrorLog.Notice("Error sending data via IP: {0}", e.Message);
                }
            }
        }

        void Parse()
        {
            while (IncomingDataBuffer.Contains("\x0A"))
            {
                string toParse = IncomingDataBuffer.Substring(0, IncomingDataBuffer.IndexOf("\n") + 1);
                _incomingdatabuffer = _incomingdatabuffer.Remove(_incomingdatabuffer.IndexOf(toParse), toParse.Length);
                Regex ToMatch = new Regex(@"(?<name>.+)\s=>\s(?<value>[^\n\r]+)[\n\r\s]+");
                Match M = ToMatch.Match(toParse);
                //ErrorLog.Notice(toParse);
                if (M.Success)
                {
                    switch (M.Groups["name"].Value)
                    {
                        case ("StreamerApp"):
                            {
                                StreamerAppVersion = M.Groups["value"].Value;
                                break;
                            }
                        case ("DeviceID"):
                            {
                                DeviceID = Convert.ToUInt16(M.Groups["value"].Value);
                                break;
                            }
                        case ("FX3"):
                            {
                                FX3Version = M.Groups["value"].Value;
                                break;
                            }
                        case ("FPGA"):
                            {
                                FPGAVersion = M.Groups["value"].Value;
                                break;
                            }
                        case ("KEYPAD"):
                            {
                                Keypad = M.Groups["value"].Value;
                                break;
                            }
                        case ("BLACK"):
                            {
                                if (M.Groups["value"].Value.Contains("disabled"))
                                {
                                    BlackoutState = false;
                                }
                                else
                                {
                                    BlackoutState = true;
                                }
                                break;
                            }
                        case ("IP"):
                            {
                                IPAddress = M.Groups["value"].Value;
                                break;
                            }
                        case ("MAC"):
                            {
                                MACAddress = M.Groups["value"].Value;
                                break;
                            }
                        case ("NETMASK"):
                            {
                                SubnetMask = M.Groups["value"].Value;
                                break;
                            }
                        case ("VIEW"):
                            {
                                CurrentLayout = (Layout)Enum.Parse(typeof(Layout), M.Groups["value"].Value,true);
                                break;
                            }
                        case ("ACFG"):
                            {
                                ACFG = Convert.ToUInt16(M.Groups["value"].Value);
                                break;
                            }
                        default: // returns that have numbers in the response type that need parsing...
                            {
                                if (M.Groups["name"].Value.Contains("Input ") && M.Groups["name"].Value.Length == 7)
                                {
                                    CurrentResolutions[Convert.ToUInt16(M.Groups["name"].Value.Substring(6,1))] = M.Groups["value"].Value;
                                }
                                else if (M.Groups["name"].Value.Contains("AIN") && M.Groups["name"].Value.Length == 4)
                                {
                                    AIN[Convert.ToUInt16(M.Groups["name"].Value.Substring(3,1))] = Utils.UndoTwosComplement(Convert.ToByte(M.Groups["value"].Value));
                                }
                                else if (M.Groups["name"].Value.Contains("AOUT") && M.Groups["name"].Value.Length == 5)
                                {
                                    AOUT[Convert.ToUInt16(M.Groups["name"].Value.Substring(4,1))] = Utils.UndoTwosComplement(Convert.ToByte(M.Groups["value"].Value));
                                }
                                else if (M.Groups["name"].Value.Contains("INV") && M.Groups["name"].Value.Length == 4)
                                {
                                    InversionStateFB[Convert.ToUInt16(M.Groups["name"].Value.Substring(3,1))] = GetInversionState(Convert.ToUInt16((M.Groups["value"].Value)));
                                }
                                break;
                            }
                    }
                }
                else if (toParse == "ACK\x0D\x0A")
                {
                    //Get(); // some command succeeded.
                }
                else if (toParse == "NACK\x0D\x0A")
                {
                    //Get(); // some command failed.
                }
                if (_incomingdatabuffer.Length > 1000)
                {
                    _incomingdatabuffer = "";
                    ErrorLog.Notice("Had to clear buffer...");
                }
            }
        }

        InversionState GetInversionState(ushort value)
        {
            switch (value)
            {
                case (1): { return InversionState.VerticalFlip; }
                case (2): { return InversionState.HorizontalFlip; }
                default: { return InversionState.NoFlip; }
            }
        }

        ushort GetInversionValue(InversionState s)
        {
            switch (s)
            {
                case (InversionState.NoFlip):
                {
                    return 0;
                }
                case(InversionState.HorizontalFlip):
                {
                    return 2;
                }
                case (InversionState.VerticalFlip):
                    {
                        return 1;
                    }
                default: { return 0; }
            }
        }

        void InvokePropertyChanged(string name, object value)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(name, value));
            }
        }

        void SetInversionStateArrayChanged(object sender, ArrayChangedEventArgs<InversionState> args)
        {
            SendCommand("INV" + Separator + args.index.ToString() + Separator + GetInversionValue(args.value).ToString());
            Thread T = new Thread(GetAsync, null);
        }

        void InversionStateFBArrayChanged(object sender, ArrayChangedEventArgs<InversionState> args)
        {
            PropertyChanged(this, new PropertyChangedEventArgs("InversionStateFB[" + args.index.ToString() + "]", GetInversionValue(args.value).ToString()));
        }
        void CurrentResolutionArrayChanged(object sender, ArrayChangedEventArgs<string> args)
        {
            PropertyChanged(this, new PropertyChangedEventArgs("CurrentResolutions[" + args.index.ToString() + "]", args.value.ToString()));
        }

        void ZoomFactorArrayChanged(object sender, ArrayChangedEventArgs<uint> args)
        {
            _zoomfactor[args.index] = args.value;
            SetZoomValue((ushort)args.index, (ushort)args.value);
        }

        object GetAsync(object o)
        {
            Crestron.SimplSharpPro.CrestronThread.Thread.Sleep(500);
            Get();
            return null;
        }
    }
    public class SHARE2
    {
        public event EventHandler<PropertyChangedEventArgs> PropertyChanged;

        private string _ipaddress;
        public string IPAddress
        {
            get { return (_ipaddress); }
            set
            {
                _ipaddress = value;
                InvokePropertyChanged("IPAddress", _ipaddress);
            }
        }

        private string _macaddress;
        public string MACAddress
        {
            get
            {
                return _macaddress;
            }
            set
            {
                _macaddress = value;
                InvokePropertyChanged("MACAddress", _macaddress);
            }
        }

        private string _subnetmask;
        public string SubnetMask
        {
            get
            {
                return _subnetmask;
            }
            set
            {
                _subnetmask = value;
                InvokePropertyChanged("SubnetMask", _subnetmask);
            }
        }

        //continue with this event invocation later...

        public void SimulateResponse(string input)
        {
            if (input.Length > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + input;
            }
        }

        public ConnectionType ConnectionType;
        public ComPort SerialPort;
        TCPClient Client;

        CTimer PollTimer;
        bool PollTimerIsTicking;

        bool _enableregularpolling;
        public bool EnableRegularPolling
        {
            get
            {
                return _enableregularpolling;
            }
            set
            {
                switch (value)
                {
                    case(true):
                    {
                        if (RegularPollInterval > 0)
                        {
                            if (_enableregularpolling == false)
                            {
                                _enableregularpolling = true;
                                RegularPoll(_enableregularpolling);
                            }
                        }
                        break;
                    }
                    default:
                    {
                        _enableregularpolling = false;
                        PollTimerIsTicking = false;
                        if (PollTimer != null)
                        {
                            PollTimer.Stop();
                            PollTimer.Dispose();
                        }
                        break; 
                    }
                }
            }
        }

        public long RegularPollInterval = 60000;
        string _incomingdatabuffer;

        public string _streamerappversion;
        public string StreamerAppVersion
        {
            get
            {
                return _streamerappversion;
            }
            private set
            {
                _streamerappversion = value;
                InvokePropertyChanged("StreamerApp", value);
            }
            
        }

        public ushort DeviceID;
        public string FX3Version;
        public string FPGAVersion;
        public string Keypad;

        bool _blackoutstate;
        public bool BlackoutState
        {
            get
            {
                return _blackoutstate;
            }
            private set
            {
                _blackoutstate = value;
                InvokePropertyChanged("Blackout", value);
            }
        }

        public sbyte[] AIN;
        public sbyte[] AOUT;
        public uint ACFG;

        uint[] _zoomfactor;
        public PropertyArray<uint> ZoomFactor;


        InversionState[] _setinversionstate;
        InversionState[] _inversionstatefb;

        public PropertyArray<InversionState> SetInversionState;
        public PropertyArray<InversionState> InversionStateFB;

        public bool IsConnected { get; private set; }

        string IncomingDataBuffer
        {
            get
            {
                return _incomingdatabuffer;
            }
            set
            {
                _incomingdatabuffer = value;
                Parse();
            }
        }

        char StartPrefix;
        char EndPrefix;
        char Separator;

        public event EventHandler<DebugEventArgs> DebugInfo;
        void Debug(string info)
        {
            if (DebugInfo != null)
            {
                DebugInfo(this, new DebugEventArgs(info));
            }
        }

        static int Port = 50000;

        Layout _currentlayout;

        public Layout CurrentLayout
        {
            get
            {
                return _currentlayout;
            }
            private set
            {
                InvokePropertyChanged("CurrentLayout", value);
                _currentlayout = value;
            }
        }

        public PropertyArray<string> CurrentResolutions;

        string [] _currentresolutions;

        void ConfigureSerialPort(ref ComPort port)
        {
            SerialPort = port;
            SerialPort.SetComPortSpec(ComPort.eComBaudRates.ComspecBaudRate9600,
                ComPort.eComDataBits.ComspecDataBits8,
                ComPort.eComParityType.ComspecParityNone,
                ComPort.eComStopBits.ComspecStopBits1,
                ComPort.eComProtocolType.ComspecProtocolRS232,
                ComPort.eComHardwareHandshakeType.ComspecHardwareHandshakeNone,
                ComPort.eComSoftwareHandshakeType.ComspecSoftwareHandshakeNone,
                false);
        }

        void InitializeArrays()
        {
            AIN = new sbyte[5];
            AOUT = new sbyte[5];
            _setinversionstate = new InversionState[5];
            _inversionstatefb = new InversionState[5];

            _currentresolutions = new string[5];
            CurrentResolutions = new PropertyArray<string>(_currentresolutions);

            SetInversionState = new PropertyArray<InversionState>(_setinversionstate);
            InversionStateFB = new PropertyArray<InversionState>(_inversionstatefb);

            _zoomfactor = new uint[5] {100,100,100,100,100};
            ZoomFactor = new PropertyArray<uint>(_zoomfactor);

            ZoomFactor.ArrayChanged += ZoomFactorArrayChanged;

            SetInversionState.ArrayChanged += SetInversionStateArrayChanged;
            InversionStateFB.ArrayChanged += InversionStateFBArrayChanged;

            CurrentResolutions.ArrayChanged += CurrentResolutionArrayChanged;
        }

        public SHARE2(ref ComPort serialPort)
        {
            ConnectionType = ConnectionType.RS232;
            try
            {
                ConfigureSerialPort(ref serialPort);
                IsConnected = true;
            }
            catch
            {
                IsConnected = false;
                Debug("Error initializing serial port.");
            }
            StartPrefix = '<';
            EndPrefix = '>';
            Separator = ' ';

            InitializeArrays();

            SerialPort.SerialDataReceived += OnSerialDataReceived;
        }

        void RegularPoll(object obj)
        {
            PollTimerIsTicking = false;
            if (obj.GetType() == typeof(bool))
            {
                if (Convert.ToBoolean(obj) == true)
                {
                    Get();
                    if (PollTimer != null)
                    {
                        PollTimer.Dispose();
                    }
                    PollTimer = new CTimer(RegularPoll, _enableregularpolling, RegularPollInterval);
                    PollTimerIsTicking = true;
                }
            }
        }

        void OnSerialDataReceived(ComPort ReceivingComPort, ComPortSerialDataEventArgs args)
        {
            if (args.SerialData.Length > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + args.SerialData;
            }
        }
        void OnIPDataReceived(TCPClient client, int numBytesReceived)
        {
            if (numBytesReceived > 0)
            {
                IncomingDataBuffer = IncomingDataBuffer + System.Text.Encoding.UTF8.GetString(client.IncomingDataBuffer, 0, numBytesReceived);
                if (Client.ClientStatus == SocketStatus.SOCKET_STATUS_CONNECTED)
                {
                    Client.ReceiveDataAsync(OnIPDataReceived);
                }
            }
        }

        public SHARE2(string IPAddress)
        {
            ConnectionType = ConnectionType.IP;
            Client = new TCPClient(IPAddress, Port, 65535);
            StartPrefix = '$';
            EndPrefix = '$';
            Separator = '_';
            InitializeArrays();
            Client.SocketStatusChange += OnConnectionStatusChange;
            Connect();
        }

        public SHARE2(IPAddress Address)
        {
            ConnectionType = ConnectionType.IP;
            Client = new TCPClient(new IPEndPoint(Address, Port), 65535);
            StartPrefix = '$';
            EndPrefix = '$';
            Separator = '_';
            InitializeArrays();
            Client.SocketStatusChange += OnConnectionStatusChange;
            Connect();
        }

        public SocketErrorCodes Connect()
        {
            if (Client != null)
            {
                return Client.ConnectToServer();
            }
            else
            {
                ErrorLog.Notice("Connect method not required for RS232 devices.");
                return SocketErrorCodes.SOCKET_BUFFER_NOT_ALLOCATED;
            }
        }

        public SocketErrorCodes Disconnect()
        {
            if (Client != null)
            {
                SocketErrorCodes DisconnectCode = Client.DisconnectFromServer();
                Client.Dispose();
                return DisconnectCode;
            }
            else
            {
                ErrorLog.Notice("Disconnect not used for RS232 devices.");
                return SocketErrorCodes.SOCKET_BUFFER_NOT_ALLOCATED;
            }
        }

        void OnConnect()
        {
            Client.ReceiveDataAsync(OnIPDataReceived);
            IsConnected = true;
        }

        void OnDisconnect()
        {
            IsConnected = false;
        }

        void OnConnectionStatusChange(TCPClient Client, SocketStatus status)
        {
            if (status == SocketStatus.SOCKET_STATUS_CONNECTED)
            {
                OnConnect();
            }
            else
            {
                OnDisconnect();
            }
        }

        public void Get()
        {
            SendCommand("GET");
            if (PollTimerIsTicking)
            {
                PollTimer.Reset(RegularPollInterval);
                //ErrorLog.Notice("poll timer reset.");
            }
        }
        public void Poll()
        {
            SendCommand("POLL");
        }
        public void SetLayout(Layout layout)
        {
            SendCommand(layout.ToString());
            Thread T = new Thread(GetAsync, null);
        }
        public void SwapVideo()
        {
            SendCommand("SW");
            Thread T = new Thread(GetAsync, null);
        }
        public void Blackout()
        {
            SendCommand("BLK");
            Thread T = new Thread(GetAsync, null);
        }
        public void UnBlackout()
        {
            SendCommand("SHW");
            Thread T = new Thread(GetAsync, null);
        }
        public void Reset()
        {
            SendCommand("RST");
        }

        public void SetAIN(byte input, sbyte value)
        {
            SendCommand("AIN" + input.ToString() + Separator + Utils.TwosComplement(value));
            Thread T = new Thread(GetAsync, null);
        }

        public void SetAOUT(byte output, sbyte value)
        {
            SendCommand("AOUT" + output.ToString() + Separator + Utils.TwosComplement(value));
            Thread T = new Thread(GetAsync, null);
        }

        public void SetUSBAsInput2()
        {
            SendCommand("SWUSB");
            Thread T = new Thread(GetAsync, null);
        }
        public void SetHDMIAsInput2()
        {
            SendCommand("SWHDMI");
            Thread T = new Thread(GetAsync, null);
        }

        void Pan(ushort camNumber, short panDirection)
        {
            SendCommand("PAN" + Separator + camNumber.ToString() + Separator + panDirection.ToString());
        }

        public void PanCounterClockwise(ushort camNumber)
        {
            Pan(camNumber, -1);
        }
        public void PanClockwise(ushort camNumber)
        {
            Pan(camNumber, 1);
        }
        public void StopPan(ushort camNumber)
        {
            Pan(camNumber, 0);
        }

        void Tilt(ushort camNumber, short TiltDirection)
        {
            SendCommand("TILT" + Separator + camNumber.ToString() + Separator + TiltDirection.ToString());
        }

        public void TiltDown(ushort camNumber)
        {
            Tilt(camNumber, -1);
        }
        public void TiltUp(ushort camNumber)
        {
            Tilt(camNumber, 1);
        }
        public void StopTilt(ushort camNumber)
        {
            Tilt(camNumber, 0);
        }

        public void SetZoomValue(ushort camNumber, ushort zoomFactor)
        {
            SendCommand("ZOOM" + Separator + camNumber.ToString() + Separator + zoomFactor.ToString());
        }
        public void ZoomIn(ushort camNumber, uint stepSize)
        {
            if ((ZoomFactor[camNumber] + stepSize) <= 500)
            {
                ZoomFactor[camNumber] = ZoomFactor[camNumber] + stepSize;
            }
        }
        public void ZoomOut(ushort camNumber, uint stepSize)
        {
            if ((ZoomFactor[camNumber] - stepSize) >= 100)
            {
                ZoomFactor[camNumber] = ZoomFactor[camNumber] - stepSize;
            }
        }

        void SendCommand(string Command)
        {
            if (this.ConnectionType == ConnectionType.RS232)
            {
                try
                {
                    SerialPort.Send(StartPrefix + Command + EndPrefix);
                }
                catch (ArgumentNullException)
                {
                    if (SerialPort == null)
                    {
                        Debug("Serial port used in initialization was null.");
                    }
                    if (Command == null)
                    {
                        Debug("Text passed to serial port was null.");
                    }
                }
            }
            else if (this.ConnectionType == ConnectionType.IP)
            {
                byte[] toSend = System.Text.Encoding.UTF8.GetBytes(StartPrefix + Command + EndPrefix);
                try
                {
                    Client.SendData(toSend, toSend.Length);
                }
                catch (Exception e)
                {
                    ErrorLog.Notice("Error sending data via IP: {0}", e.Message);
                }
            }
        }

        void Parse()
        {
            while (IncomingDataBuffer.Contains("\x0D"))
            {
                string toParse = IncomingDataBuffer.Substring(0, IncomingDataBuffer.IndexOf("\x0D") + 1);
                _incomingdatabuffer = _incomingdatabuffer.Remove(_incomingdatabuffer.IndexOf(toParse), toParse.Length);
                Regex ToMatch = new Regex(@"(?<name>.+)\s=>\s(?<value>[^\n\r]+)[\n\r\s]+");
                Match M = ToMatch.Match(toParse);
                //ErrorLog.Notice(toParse);
                if (M.Success)
                {
                    switch (M.Groups["name"].Value)
                    {
                        case ("StreamerApp"):
                            {
                                StreamerAppVersion = M.Groups["value"].Value;
                                break;
                            }
                        case ("DeviceID"):
                            {
                                DeviceID = Convert.ToUInt16(M.Groups["value"].Value);
                                break;
                            }
                        case ("FX3"):
                            {
                                FX3Version = M.Groups["value"].Value;
                                break;
                            }
                        case ("FPGA"):
                            {
                                FPGAVersion = M.Groups["value"].Value;
                                break;
                            }
                        case ("KEYPAD"):
                            {
                                Keypad = M.Groups["value"].Value;
                                break;
                            }
                        case ("BLACK"):
                            {
                                if (M.Groups["value"].Value.Contains("disabled"))
                                {
                                    BlackoutState = false;
                                }
                                else
                                {
                                    BlackoutState = true;
                                }
                                break;
                            }
                        case ("IP"):
                            {
                                IPAddress = M.Groups["value"].Value;
                                break;
                            }
                        case ("MAC"):
                            {
                                MACAddress = M.Groups["value"].Value;
                                break;
                            }
                        case ("NETMASK"):
                            {
                                SubnetMask = M.Groups["value"].Value;
                                break;
                            }
                        case ("VIEW"):
                            {
                                CurrentLayout = (Layout)Enum.Parse(typeof(Layout), M.Groups["value"].Value,true);
                                break;
                            }
                        case ("ACFG"):
                            {
                                ACFG = Convert.ToUInt16(M.Groups["value"].Value);
                                break;
                            }
                        default: // returns that have numbers in the response type that need parsing...
                            {
                                if (M.Groups["name"].Value.Contains("Input ") && M.Groups["name"].Value.Length == 7)
                                {
                                    CurrentResolutions[Convert.ToUInt16(M.Groups["name"].Value.Substring(6,1))] = M.Groups["value"].Value;
                                }
                                else if (M.Groups["name"].Value.Contains("AIN") && M.Groups["name"].Value.Length == 4)
                                {
                                    AIN[Convert.ToUInt16(M.Groups["name"].Value.Substring(3,1))] = Utils.UndoTwosComplement(Convert.ToByte(M.Groups["value"].Value));
                                }
                                else if (M.Groups["name"].Value.Contains("AOUT") && M.Groups["name"].Value.Length == 5)
                                {
                                    AOUT[Convert.ToUInt16(M.Groups["name"].Value.Substring(4,1))] = Utils.UndoTwosComplement(Convert.ToByte(M.Groups["value"].Value));
                                }
                                else if (M.Groups["name"].Value.Contains("INV") && M.Groups["name"].Value.Length == 4)
                                {
                                    InversionStateFB[Convert.ToUInt16(M.Groups["name"].Value.Substring(3,1))] = GetInversionState(Convert.ToUInt16((M.Groups["value"].Value)));
                                }
                                break;
                            }
                    }
                }
                else if (toParse.Contains("BLACK")) //BLACK response in <GET> does not match regex, FW bug.
                {
                    if (toParse.Contains("disabled"))
                    {
                        BlackoutState = false;
                    } 
                    else if (toParse.Contains("enabled"))
                    {
                        BlackoutState = true;
                    }
                }
                else if (toParse == "ACK\x0D")
                {
                    //Get(); // some command succeeded.
                }
                else if (toParse == "NACK\x0D")
                {
                    //Get(); // some command failed.
                }
                if (_incomingdatabuffer.Length > 1000)
                {
                    _incomingdatabuffer = "";
                    ErrorLog.Notice("Had to clear buffer...");
                }
            }
        }

        InversionState GetInversionState(ushort value)
        {
            switch (value)
            {
                case (1): { return InversionState.VerticalFlip; }
                case (2): { return InversionState.HorizontalFlip; }
                default: { return InversionState.NoFlip; }
            }
        }

        ushort GetInversionValue(InversionState s)
        {
            switch (s)
            {
                case (InversionState.NoFlip):
                {
                    return 0;
                }
                case(InversionState.HorizontalFlip):
                {
                    return 2;
                }
                case (InversionState.VerticalFlip):
                    {
                        return 1;
                    }
                default: { return 0; }
            }
        }

        void InvokePropertyChanged(string name, object value)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(name, value));
            }
        }

        void SetInversionStateArrayChanged(object sender, ArrayChangedEventArgs<InversionState> args)
        {
            SendCommand("INV" + Separator + args.index.ToString() + Separator + GetInversionValue(args.value).ToString());
            Thread T = new Thread(GetAsync, null);
        }

        void InversionStateFBArrayChanged(object sender, ArrayChangedEventArgs<InversionState> args)
        {
            PropertyChanged(this, new PropertyChangedEventArgs("InversionStateFB[" + args.index.ToString() + "]", GetInversionValue(args.value).ToString()));
        }
        void CurrentResolutionArrayChanged(object sender, ArrayChangedEventArgs<string> args)
        {
            PropertyChanged(this, new PropertyChangedEventArgs("CurrentResolutions[" + args.index.ToString() + "]", args.value.ToString()));
        }

        void ZoomFactorArrayChanged(object sender, ArrayChangedEventArgs<uint> args)
        {
            _zoomfactor[args.index] = args.value;
            SetZoomValue((ushort)args.index, (ushort)args.value);
        }

        object GetAsync(object o)
        {
            Crestron.SimplSharpPro.CrestronThread.Thread.Sleep(500);
            Get();
            return null;
        }
    }
    public enum ConnectionType : byte
    {
        RS232 = 0,
        IP = 1
    }
    public enum Layout : ushort
    {
        S1 = 1,
        S2 = 2,
        TB1 = 11,
        TB2 = 12,
        BS1 = 21,
        BS2 = 22,
        PPTR1 = 31,
        PPTR2 = 32,
        PPTL1 = 41,
        PPTL2 = 42,
        PPBR1 = 51,
        PPBR2 = 52,
        PPBL1 = 61,
        PPBL2 = 62,
        SS1 = 71,
        SS2 = 72
    }
    public enum InversionState : byte
    {
        NoFlip = 0,
        VerticalFlip = 1,
        HorizontalFlip = 2
    }
    public class DebugEventArgs : EventArgs
    {
        public string DebugString;
        public DebugEventArgs(string debugString)
        {
            DebugString = debugString;

        }
    }
    public class PropertyChangedEventArgs : EventArgs
    {
        public string PropertyName;
        public object PropertyValue;
        public PropertyChangedEventArgs(string name, object value)
        {
            PropertyName = name;
            PropertyValue = value;
        }
    }

    public static class Utils
    {
        public static byte TwosComplement(sbyte value)
        {
            if (value >= 0)
                return (byte)value;
            else
                return (byte)(256 + value);
        }
        public static sbyte UndoTwosComplement(byte value)
        {
            if ((value >= 0) && (value < 128))
                return (sbyte)value;
            else
                return (sbyte)(value - 256);
        }
    }

    public class PropertyArray<T>
    {
        T[] BaseArray;
        public event EventHandler<ArrayChangedEventArgs<T>> ArrayChanged;

        public int Length
        {
            get
            {
                return BaseArray.Length;
            }
        }
        public T this[int index]
        {
            get
            {
                return BaseArray[index];
            }
            set
            {
                BaseArray[index] = value;
                ArrayChanged.Invoke(this, new ArrayChangedEventArgs<T>(index, value));
            }
        }

        public PropertyArray(T[] array)
        {
            BaseArray = array;         
        }
    }

    public class ArrayChangedEventArgs<T> : EventArgs
    {
        public int index;
        public T value;
        public ArrayChangedEventArgs(int changedIndex, T newValue)
        {
            index = changedIndex;
            value = newValue;
        }
    }
}

